/* Copyright (c) 2004, 2009, Oracle and/or its affiliates. 
All rights reserved. */

/*
 * NAME
 * macrodemo.cpp: Demonstrates agent with inbuilt unit test macros
 *
 * DESCRIPTION
 * This is a sample program that demonstrates an agent to monitor
 * a file. The agent has the following tasks:
 *  - On startup         : Create the file.
 *  - On shutdown        : Delete the file.
 *  - On check command   : Detect whether the file is present or not.
 *  - On clean command   : Delete the file.
 * The embedded unit test macros can be used to test the agent functionality
 * even without the presence of Clusterware
 *
 * RELATED DOCUMENTS
 *  (1) readme.txt, which should be available in the same directory as this file.
 *  (2) demoagent1.cpp, a C++ agent which monitors a file.
 *  (3) demoagent2.cpp, a script/C++ hybrid agent to also monitor a file
 *  (4) clsagfwm.h, the header which defines the test macro API
 *  (4) The Oracle Clusterware Administration and Deployment Guide - 11g Release 2
 *  (5) The Oracle Technology Network - available online at http://www.oracle.com/technology
 *
 * NOTES
 *  (1) The clsagfw_log messages are sent by default to the agent log directory. This 
 *      would typically be $(INSTALL_DIR)/log/<hostname>/agent/crsd/
 *  (2) The file to be monitored is determined by reading the PATH_NAME attribute for the
 *      resource. This attribute is set when the resource is defined. Look at the readme 
 *      file for more information.
 *  
 * MODIFIED
 *    skakarla   05/20/09 - adding log level
 *    vmanivel   03/25/09 - Updating based on comments
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <clsagfw.h>
#include <clsagfwm.h>

#define TEST_TYPE1                   (oratext *) "test_type1"
#define TEST_RES1                    (oratext *) "res1"
#define TESTTYPE_A_PATHNAME          (oratext *) "PATH_NAME"


/*
 * UNIT TEST ACTIONS
 * Agent framework reads the actions mentioned in below table and
 * executes them when ever the agent starts with "-t <testname>" argument.
 * Using this infrastructure agent can be unit tested as a stand alone
 * program.
 */
AGFW_BEGINTEST_ACTIONS(g_testActions)
    /*
     * Add test type with one attribute pathName.
     */
    AGFW_ADDTYPE_BEGIN((char *)TEST_TYPE1)
    AGFW_ADD_STRATTR((char *)TESTTYPE_A_PATHNAME, 
		     "%ORA_CRS_HOME%/testtype.txt")
    AGFW_ADDTYPE_END()

    /*
     * Add resource of the specified type
     * Modify the attribute pathname and timeout.
     */
    AGFW_ADDRES_BEGIN((char *)TEST_TYPE1, (char *)TEST_RES1)
    AGFW_MODIFY_STRATTR((char *)TESTTYPE_A_PATHNAME, 
			"%ORA_CRS_HOME%/res1.txt")
    AGFW_MODIFY_INTATTR("START_TIMEOUT", "0")
    AGFW_ADDRES_END()

    /*
     * Add actions to this test - start/stop the resource
     */
    AGFW_START_RESOURCE((char *)TEST_RES1)
    AGFW_STOP_RESOURCE((char*) TEST_RES1)


AGFW_ENDTEST_ACTIONS()


  /*
   * Now add the tests defined above to the test array
   * These tests can be invoked by name later.
   * Since this test has only 1 set of tests, 
   * demoagent -t is sufficient, and the first and only test is run.
   */
AGFW_BEGINTEST_ARRAY(g_allTests)  

  AGFW_ADDTEST((oratext *)"dummy", (clsagfw_testentry *)g_testActions)

AGFW_ENDTEST_ARRAY()

/*
 * NAME 
 *    test_agent_exit
 * DESCRIPTION
 *    Function to clean up, called by framework whenever it receives 
 * the AGENT_EXIT message from crsd.
 * PARAMS
 *    exitCode, which is an sb4 describing the reason for exit.
 * RETURNS
 *    void
 */
void 
test_agent_exit(sb4 exitCode)
{
  clsagfw_log(NULL, 1, (oratext *)"Demo Agent is exiting..\n");
}


/*
 * NAME
 *    type1_start
 * DESCRIPTION
 *    Action entry point for the 'start' command. Creates the file for monitoring.
 * PARAMS
 *    Pointer to a clsagfw struct. This is created and deleted internally by the
 *    framework and is opaque to the agent. Do not modify it.
 * RETURNS
 *    ub4 which could be CLSAGFW_AE_FAIL or CLSAGFW_AE_SUCCESS depending on whether
 *    the action failed or succeeded.
 */
ub4 
type1_start(const clsagfw_aectx *ectx)
{
    ub4  ret = CLSAGFW_AE_FAIL;
    const oratext   *pResName = NULL;
    const oratext   *pPathName = NULL;
    char            cmdBuff[256];

    clsagfw_log(ectx, 1, (oratext *)"Start action called..");
    if (clsagfw_get_attrvalue(ectx, (oratext *)"NAME", &pResName) != 
            CLSAGFW_SUCCESS)
    {
        goto done;
    }

    if (clsagfw_get_attrvalue(ectx, TESTTYPE_A_PATHNAME, &pPathName) !=
            CLSAGFW_SUCCESS)
    {
        pPathName = pResName;
    }

    clsagfw_log(ectx, 1, 
                (oratext *)"Start action arguments: resName = %s, pathName = %s", 
                pResName, pPathName);

    sprintf(cmdBuff, "touch %s", pPathName);
    if (system(cmdBuff) != 0)
    {
        printf("START action for resource %s: FAILED\n", pResName);
        ret = CLSAGFW_AE_FAIL;
    }
    else
    {
        printf("START action for resource %s: SUCCESS\n", pResName);
        ret = CLSAGFW_AE_SUCCESS;
    }

done:
    return ret;
}

/*
 * NAME
 *    type1_stop
 * DESCRIPTION
 *    Action entry point for the 'stop' and 'clean' commands. Deletes the file being monitored.
 * PARAMS
 *    Pointer to a clsagfw struct. This is created and deleted internally by the
 *    framework and is opaque to the agent. Do not modify it.
 * RETURNS
 *    ub4 which could be CLSAGFW_AE_FAIL or CLSAGFW_AE_SUCCESS depending on whether
 *    the action failed or succeeded.
 */
ub4 
type1_stop (const clsagfw_aectx *ectx)
{
    ub4             ret = CLSAGFW_AE_FAIL;
    const oratext   *pResName = NULL;
    const oratext   *pPathName = NULL;
    char            cmdBuff[256];


    clsagfw_log(ectx, 1, (oratext *)"Stop action called..");
    if (clsagfw_get_attrvalue(ectx, (oratext *)"NAME", &pResName) != 
            CLSAGFW_SUCCESS)
    {
        goto done;
    }

    if (clsagfw_get_attrvalue(ectx, TESTTYPE_A_PATHNAME, &pPathName) !=
            CLSAGFW_SUCCESS)
    {
        pPathName = pResName;
    }

    clsagfw_log(ectx, 1, 
                (oratext *)"Stop action arguments: resName = %s, pathName = %s",
                pResName, pPathName);

    sprintf(cmdBuff, "rm -f %s", pPathName);
    if (system(cmdBuff) != 0)
    {
        printf("STOP action for resource %s: FAILED\n", pResName);
        ret = CLSAGFW_AE_FAIL;
    }
    else
    {
        printf("STOP action for resource %s: SUCCESS\n", pResName);
        ret = CLSAGFW_AE_SUCCESS;
    }

done:
    return ret;
}

/*
 * NAME
 *    type1_check
 * DESCRIPTION
 *    Action entry point for the 'check' command. Determines if the file exists.
 * PARAMS
 *    Pointer to a clsagfw struct. This is created and deleted internally by the
 *    framework and is opaque to the agent. Do not modify it.
 * RETURNS
 *    ub4 which gives the status of the resource. Look at clsagfw.h for details on
 *    this function's return codes.
 */
ub4 
type1_check(const clsagfw_aectx *ectx)
{
    ub4             ret = CLSAGFW_UNKNOWN;
    const oratext   *pResName = NULL;
    const oratext   *pPathName = NULL;
    char            cmdBuff[256];

    clsagfw_log(ectx, 1, (oratext *)"Check action called..");

    if (clsagfw_get_attrvalue(ectx, (oratext *) "NAME", &pResName) != 
            CLSAGFW_SUCCESS)
    {
        goto done;
    }

    if (clsagfw_get_attrvalue(ectx, TESTTYPE_A_PATHNAME, &pPathName) !=
            CLSAGFW_SUCCESS)
    {
        pPathName = pResName;
    }

    clsagfw_log(ectx, 1, 
                (oratext *)"Check action arguments: resName = %s, pathName = %s", 
                pResName, pPathName);

    sprintf(cmdBuff, "ls %s", pPathName);
    if (system(cmdBuff) != 0)
    {
        printf("CHECK action: %s status - UNPLANNED_OFFLINE\n", pResName);
        ret = CLSAGFW_UNPLANNED_OFFLINE;
    }
    else
    {
        printf("CHECK action: %s status - ONLINE\n", pResName);
        ret = CLSAGFW_ONLINE;
    }

done:
    return ret;
}



/*
 * NAME
 *    add_type_to_agfw
 * DESCRIPTION
 *    Adds the specified type to the agent framework and registers the action
 *    entry points.
 * PARAMS
 *    The typename to be added to the framework
 * RETURNS
 *    void
 */
void
add_type_to_agfw(const oratext *typeName)
{
    if (clsagfw_add_type(typeName) != CLSAGFW_SUCCESS)
    {
        printf("Failed in adding type %s to the framework\n", typeName);
        clsagfw_exit(-1);
    }

    /*
     * Set all call back action entries for TYPE-1
     */
    clsagfw_set_entrypoint(typeName, type1_start, CLSAGFW_ACTION_START);
    clsagfw_set_entrypoint(typeName, type1_stop, CLSAGFW_ACTION_STOP);
    clsagfw_set_entrypoint(typeName, type1_check, CLSAGFW_ACTION_CHECK);
    clsagfw_set_entrypoint(typeName, type1_stop, CLSAGFW_ACTION_CLEAN);
    printf ("Added resource type [%s] to the agent framework\n", typeName);
}


/*
 *  Initialization of the agent framework is done in main
 */
int main(sb4 argc, oratext **argv)
{
    printf(" *** Agent Framework Demo Agent Started *** \n");

    /*
     * Initialize the agent framework 
     */ 
    if (clsagfw_init(argc, argv, 0, NULL, (void *)g_allTests) 
        != CLSAGFW_SUCCESS)
    {
        printf("Failed to initilize the agent framework\n");
        clsagfw_exit(-1);
    }

    clsagfw_set_exitcb(test_agent_exit);

    add_type_to_agfw(TEST_TYPE1);

    /*
     * All set to go, Start the framework
     * this function is a blocking call, it would return under only
     * one condition, that is when ever CRSD sends an AGENT_EXIT
     * command
     */
    clsagfw_startup();

    /*** NOT REACHED **/

    return 0; 
}


